//	Maze4DMaze.swift
//
//	© 2025 by Jeff Weeks
//	See TermsOfUse.txt

import SwiftUI


enum DifficultyLevel: Int, CaseIterable, Identifiable {

	case easy		= 2	//	2×2×2×2 lattice of nodes
	case hard		= 3	//	3×3×3×3 lattice of nodes
	case extraHard	= 4	//	4×4×4×4 lattice of nodes
	
	var id: Int { self.rawValue }

	var  n: Int { self.rawValue }
	
	var localizedNameKey: LocalizedStringKey {
	
		switch self {
		case .easy:			return "NewMaze-Size2"
		case .hard:			return "NewMaze-Size3"
		case .extraHard:	return "NewMaze-Size4"
		}
	}
	
	var color: Color {
	
		switch self {
		case .easy:			return Color(.displayP3, red: 0.0, green: 1.0, blue: 0.5)
		case .hard:			return Color(.displayP3, red: 1.0, green: 0.9, blue: 0.0)
		case .extraHard:	return Color(.displayP3, red: 1.0, green: 0.0, blue: 0.0)
		}
	}
}

struct MazeNodeEdges {

	//	Does an edge connect this node to the next node
	//	in the positive x, y, z or w direction?
	var outbound: [Bool]

	//	Does an edge come in from the previous node
	//	in the negative x, y, z or w direction?
	var inbound: [Bool]
}

//	Each node in the n × n × n × n lattice may be
//	referred to by a quadruple (i_x, i_y, i_z, i_w) of integers,
//	each in the range {0, 1, … , n-1}.
typealias MazeNodeIndex = SIMD4<Int>

enum SliderPosition {

	case atNode(
		nodeIndex: MazeNodeIndex)
	
	case onOutboundEdge(
		baseNodeIndex: MazeNodeIndex,
		direction: Int,		//	which axis the slider is displaced along
							//		∈ {0, 1, 2, 3}
		distance: Double)	//	how far the slider is from the base node,
							//		as a fraction of the distance to the next node
							//		0.0 ≤ distance ≤ 1.0
}

struct MazeData {	//	Let's call this struct "MazeData", not "Maze",
					//	so we can search for all occurrences more easily.

	let difficultyLevel: DifficultyLevel
	
	let edges: [ [ [ [ MazeNodeEdges ] ] ] ]	//	array of size n × n × n × n

	var sliderPosition: SliderPosition
	var sliderCoastingSpeed: Double?	//	in tube-lengths per second

	let goalPosition:   MazeNodeIndex
}


// MARK: -
// MARK: Maze maker

func makeNewMaze(
	_ difficulty: DifficultyLevel
) -> MazeData {

	let theDifficultyLevel = difficulty

	let theEdges = makeMazeEdges(mazeSize: difficulty.n)
	
	let (theSliderStartNodeIndex, theGoalNodeIndex)
		= positionSliderAndGoal(edges: theEdges)

	let theMaze = MazeData(
		difficultyLevel: theDifficultyLevel,
		edges: theEdges,
		sliderPosition: .atNode(nodeIndex: theSliderStartNodeIndex),
		sliderCoastingSpeed: nil,
		goalPosition: theGoalNodeIndex)

	return theMaze
}

func makeMazeEdges(
	mazeSize n: Int
) -> [ [ [ [ MazeNodeEdges ] ] ] ] {	//	array of size n × n × n × n

	//	Based on the algorithm in the Maze program written
	//	by Jeff Weeks for Adam Weeks Marano, Christmas 1992.

	//	The plan is to give each node its own index.
	//	We then put all the possible outbound edges on a list,
	//	randomize the order of the list,
	//	and then go down it one item at a time.
	//	Whenever a possible edge connects two nodes with different indices,
	//	add the edge to the maze and merge the neighboring indices
	//	(that is, set all occurrences of one index to equal the value
	//	of the other index).  When a potential edge connects two nodes
	//	of the same index, omit it from the maze.  It's easy to see that
	//	this will yield a maze with a unique path between any two nodes.
	
	//	Initialize all maze edges to false.
	var theEdges = Array(
		repeating: Array(
			repeating: Array(
				repeating: Array(
					repeating:
						MazeNodeEdges(
							outbound: [false, false, false, false],
							inbound:  [false, false, false, false]),
					count: n),
				count: n),
			count: n),
		count: n)

	//	Initialize thePotentialEdgeList.
	struct PotentialEdge {
		let startingNodeI: Int	//	starting node's horizontal index
		let startingNodeJ: Int	//	starting node's vertical index
		let startingNodeK: Int	//	starting node's depth index
		let startingNodeL: Int	//	starting node's ana index
		let direction: Int		//	0, 1, 2 or 3, meaning direction h, v, d or a
	}
	var thePotentialEdgeList: [PotentialEdge] = []
	thePotentialEdgeList.reserveCapacity(n * n * n * n * 4)	//	slightly more than necessary
	for i in 0 ..< n {
		for j in 0 ..< n {
			for k in 0 ..< n {
				for l in 0 ..< n {
					for d in 0 ... 3 {
					
						//	This is a bounded maze, not a torus maze.
						//	So the last nodes in any given direction
						//	never have an outbound edge in that same direction.
						if (d == 0 && i == n - 1)
						|| (d == 1 && j == n - 1)
						|| (d == 2 && k == n - 1)
						|| (d == 3 && l == n - 1) {
							continue
						}

						let thePotentialEdge = PotentialEdge(
												startingNodeI: i,
												startingNodeJ: j,
												startingNodeK: k,
												startingNodeL: l,
												direction: d)
												
						thePotentialEdgeList.append(thePotentialEdge)
					}
				}
			}
		}
	}
	
	//	Randomize thePotentialEdgeList.
	thePotentialEdgeList.shuffle()

	//	Initialize the node tags to arbitrary but distinct values.
	var theNodeTags = Array(
						repeating: Array(
							repeating: Array(
								repeating: Array(
									repeating: 0,
									count: n),
								count: n),
							count: n),
						count: n)
	var theNodeTag = 0
	for i in 0 ..< n {
		for j in 0 ..< n {
			for k in 0 ..< n {
				for l in 0 ..< n {
					theNodeTags[i][j][k][l] = theNodeTag
					theNodeTag += 1
				}
			}
		}
	}

	//	Go down aPotentialEdgeList and add an edge whenever
	//	it connects two nodes with different tags.
	for thePotentialEdge in thePotentialEdgeList {
	
		let i0 = thePotentialEdge.startingNodeI
		let j0 = thePotentialEdge.startingNodeJ
		let k0 = thePotentialEdge.startingNodeK
		let l0 = thePotentialEdge.startingNodeL
		let d  = thePotentialEdge.direction
		
		var i1 = i0
		var j1 = j0
		var k1 = k0
		var l1 = l0
		switch d {
		case 0: i1 += 1
		case 1: j1 += 1
		case 2: k1 += 1
		case 3: l1 += 1
		default: preconditionFailure("Invalid direction in makeMazeEdges()")
		}
		
		let theTag0 = theNodeTags[i0][j0][k0][l0]
		let theTag1 = theNodeTags[i1][j1][k1][l1]
		
		if theTag0 != theTag1 {
		
			//	Replace all occurrences of theTag1 with theTag0.
			for i in 0 ..< n {
				for j in 0 ..< n {
					for k in 0 ..< n {
						for l in 0 ..< n {
							if theNodeTags[i][j][k][l] == theTag1 {
								theNodeTags[i][j][k][l] = theTag0
							}
						}
					}
				}
			}

			//	Add the edge to the maze.
			theEdges[i0][j0][k0][l0].outbound[d] = true
			theEdges[i1][j1][k1][l1].inbound[d]  = true
		}
	}

	return theEdges
}

func positionSliderAndGoal(
	edges: [ [ [ [ MazeNodeEdges ] ] ] ]	//	array of size n × n × n × n
) -> (MazeNodeIndex, MazeNodeIndex) {

	//	Proposition
	//
	//	Let node A be an arbitrary node in the maze.
	//	Let node B be a node maximally far from node A
	//		(following the maze, of course).
	//	Let node C be a node maximally far from node B
	//		(following the maze, of course).
	//	Then the distance from node B to node C is
	//		greater than or equal to the distance between
	//		any other pair of nodes.
	//
	//	The proof is easy:  Imagine the maze's edges to be pieces
	//	of string, all of equal length, tied together at the nodes.
	//	Grab node A and hoist it up into the air, letting the rest
	//	of the maze dangle below it.  Let node B be one of the
	//	maximally low nodes.  Now let go of node A and instead
	//	grab node B and hoist it up into the air.  Let node C be
	//	one of the maximally low nodes.  Nodes B and C are
	//	maximally far apart (among all possible pairs of nodes
	//	in the whole maze).  There's no need to keep repeating
	//	the hoisting procedure.  To rigorously prove this proposition,
	//	imagine some maximally distant pair of nodes, color
	//	the sequences of strings connecting them red, and trace
	//	through what happens to the red string as you carry out
	//	the procedure just described (the details aren't difficult,
	//	but neither are they so brief that I feel like writing them all down).
	//	Presumably this result appears somewhere in the literature,
	//	even though I couldn't find it on the web (and Patti Lock
	//	was unaware of it).
	
	let nodeA = MazeNodeIndex(0, 0, 0, 0)

	//	Find a pair of maximally distant points in the maze.
	let nodeB = furthestNode(from: nodeA, inMaze: edges)
	let nodeC = furthestNode(from: nodeB, inMaze: edges)

	return (nodeB, nodeC)
}

func furthestNode(
	from baseNode: MazeNodeIndex,
	inMaze edges: [ [ [ [ MazeNodeEdges ] ] ] ]	//	array of size n × n × n × n
) -> MazeNodeIndex {	//	returns index of a node maximally far from the baseNode

	struct MazeTreeNode {
		let thisNode: MazeNodeIndex
		let parentNode: MazeNodeIndex
	}
	let theInvalidNode = MazeNodeIndex(Int.max, Int.max, Int.max, Int.max)	//	easier than using an optional

	let n = edges.count
	
	var theQueue: [MazeTreeNode] = Array(
		repeating: MazeTreeNode(
			thisNode: MazeNodeIndex(0, 0, 0, 0),
			parentNode: MazeNodeIndex(0, 0, 0, 0)),
			count: n * n * n * n)
	
	theQueue[0] = MazeTreeNode(thisNode: baseNode, parentNode: theInvalidNode)
	var theQueueStart = 0
	var theQueueEnd = 0

	while theQueueStart <= theQueueEnd {
	
		//	Pull the next node off the queue.
		let theNode = theQueue[theQueueStart]
		theQueueStart += 1
		
		let theNodeIndex = theNode.thisNode
		let theParentIndex = theNode.parentNode

		//	Add each of theNode's accessible neighbors,
		//	excluding its own parent, to the queue.
		let theNodeEdges = edges[theNodeIndex[0]][theNodeIndex[1]][theNodeIndex[2]][theNodeIndex[3]]
		for theDirection in 0...3 {
		
			if theNodeEdges.outbound[theDirection] {
			
				var theNeighborIndex = theNodeIndex
				theNeighborIndex[theDirection] += 1
				
				if theNeighborIndex != theParentIndex {
				
					theQueueEnd += 1
					theQueue[theQueueEnd] = MazeTreeNode(
												thisNode: theNeighborIndex,
												parentNode: theNodeIndex)
				}
			}
		
			if theNodeEdges.inbound[theDirection] {
			
				var theNeighborIndex = theNodeIndex
				theNeighborIndex[theDirection] -= 1
				
				if theNeighborIndex != theParentIndex {
				
					theQueueEnd += 1
					theQueue[theQueueEnd] = MazeTreeNode(
												thisNode: theNeighborIndex,
												parentNode: theNodeIndex)
				}
			}
		}
	}
	
	precondition(
		theQueueStart == n * n * n * n,
		"Wrong number of elements processed on queue in furthestNode()")

	//	We did a breadth first traversal of the tree,
	//	so the last node considered must be maximally far from the first.
	//
	let theFurthestNode = theQueue[theQueueEnd].thisNode

	return theFurthestNode
}


// MARK: -
// MARK: Slider and goal positions

func sliderOffsetCenterAndColor(
	sliderPosition: SliderPosition,
	difficulty: DifficultyLevel,
	shearFactor: Double
) -> (SIMD3<Double>, simd_half3) {	//	(center, color)

	let p4D: SIMD4<Double>
	switch sliderPosition {
	
	case .atNode(let nodeIndex):
	
		p4D = SIMD4<Double>(nodeIndex)
		
	case .onOutboundEdge(let theBaseNodeIndex, let theDirection, let theDistance):
	
		//	Compute the base node's center.
		var thePosition = SIMD4<Double>(theBaseNodeIndex)
		
		//	Move theSliderCenter along the outbound edge
		//	through the given distance in the given direction.
		thePosition[theDirection] += theDistance
		
		p4D = thePosition
	}

	let (theSliderCenter, _, theSaturatedColor) = shearedPosition(
													position4D: p4D,
													difficulty: difficulty,
													shearFactor: shearFactor)

	let theWhiteColor = simd_half3(1.0, 1.0, 1.0)
	let thePastelColor = Float16(0.5) * theSaturatedColor
					   + Float16(0.5) * theWhiteColor

	return (theSliderCenter, thePastelColor)
}


func shearedPosition(
	position4D: SIMD4<Double>,
	difficulty: DifficultyLevel,
	shearFactor: Double
) -> (SIMD3<Double>, Double, simd_half3) {	//	(sheared position, hue, saturated RGB)

	//	To distinguish nodes whose x-, y- and z-coordinates all agree,
	//	we slightly shear the 4D space before projecting down to 3D.
	
	//	Let the amount of shear between two adjacent levels (differing by Δw = ±1)
	//	be the tubeRadius times a user-controllable shear factor.
	//
	let theShearPerLevel = shearFactor * tubeRadius
	
	//	The total shear is amount of shear between
	//	the kata-most level and the ana-most level.
	//
	let theMaxW = Double(difficulty.n - 1)
	let theTotalShear = theShearPerLevel * theMaxW
	
	//	What fraction of the way does the position4D sit
	//	between w = 0 and w = theMaxW ?
	//
	let theWFraction = position4D[3] / theMaxW

	//	As theWFraction runs from 0.0 to 1.0,
	//	let the offset run from -theTotalShear/2 to +theTotalShear/2
	//
	let theOffset = (-0.5 + theWFraction) * theTotalShear

	//	Offset the position equally in the x, y and z directions.
	//
	let theShearedPosition3D = SIMD3<Double>(
		position4D[0] + theOffset,
		position4D[1] + theOffset,
		position4D[2] + theOffset
	)

	//	As theWFraction runs from 0.0 to 1.0,
	//	let the hue run from hueKata to hueAna.
	//
	let theHue = hueKata + theWFraction * (hueAna - hueKata)
	let theSaturatedColor = HSVtoRGB(hue: theHue, saturation: 1.0, value: 1.0)

	return (theShearedPosition3D, theHue, theSaturatedColor)
}

func sliderHasReachedGoal(
	maze: MazeData?
) -> Bool {

	guard let theMaze = maze else {
		assertionFailure("Internal error:  maze is nil in sliderHasReachedGoal()")
		return false
	}

	switch theMaze.sliderPosition {
	
	case .atNode(nodeIndex: let theSliderNodeIndex):
		
		return theSliderNodeIndex == theMaze.goalPosition
		
	case .onOutboundEdge(baseNodeIndex: _, direction: _, distance: _):

		return false
	}
}
